
# -*- coding: utf-8 -*-

#### Importation des bibliothèques utiles

import numpy as np
import matplotlib.pyplot as plt
from scipy.integrate import odeint

#### Initialisations

"""
Résolution numérique d'une équation différentielle d'ordre 2
avec la fonction odeint de python (scipy).
v2(t) tension d'entrée du non inverseur = sortie de Wien
v1(t) tension de sortie du non inverseur = entrée de Wien
Equadiff v1"(t) + (wo/Q)v1'(t) + wo^2 v(t) = Ho(wo/Q)v'(t)
en régime linéaire v2(t) = Gv1(t) : v1"(t) + wo / Q (1- HoG)v1'(t) + wo^2 v1(t) = 0
en regime saturé v2(t) = +/- Vsat : v1"(t) + (wo/Q)v1'(t) + wo^2 v(t) = 0
"""

# wo pulsation propre en rad/s
fo = 1600    #Hz
wo = 2*np.pi*fo

# filtre passe-bande
Q = 1/3       # facteur de qualité
Ho = 1/3


GHo = 1.05

# Gain du non inverseur en régime linéaire
G = 3*GHo 

Vsat = 15    # V

# en régime linéaire
tau = - (2*Q)/(wo*(1 - GHo))   # constante de temps de l'exponentielle
print (" constante de temps de l'exponentielle en régime linéaire tau = {:.3E} s".format(tau))

# Conditions initiales
v10 = 0.01                # 0.001 V tension d'offset ALI
vp0 = 0                     
y0 = [v10, vp0]           # vecteur initial, tension et dérivée première

# équa diff V' = f(V,t)
def f(V,t) :             # mélange de lin et sat
    v1, vp = V
    if np.abs(v1)<Vsat/G :
        vs = - wo*(1 - GHo)*vp/Q -v1*wo**2 
    else :
        vs = - wo*vp/Q -v1*wo**2
    return [vp, vs]                  # renvoie la liste V'= [v', v"]

# Données pour l'expérience numérique
n = 10000                    # nombre de points de l'échantillonnage
t0 = 0                      # instant initial en s
t1 = 0.015                   # instant final en s

#### Expérience numérique
t = np.linspace(t0, t1, n)
sol = odeint(f, y0, t)      # solution = liste des vecteurs pour chaque t

Lv1 = sol[:,0]    # on extrait v1 de sol

Lv2 = G*Lv1
for i in range(n) :
    if np.abs(Lv2[i])> Vsat :
        Lv2[i] = Vsat*np.sign(Lv2[i])



#### Tracés
# v(t) 
plt.figure(1)                                   # fenetrage
plt.axes(xlim = (t[0],t[-1]),ylim = (-Vsat*1.1, Vsat*1.1))
plt.plot(t, Lv2, label= f'v2(t) GHo = {GHo}')      #plt. instruction dans matplotlib.pyplot
plt.plot(t, Lv1, label='v1(t)')                 # syntaxe plot(liste des abscisses,liste ordonnées,...)
                                                # sol = liste des vecteurs solution
                                                # sol[:, 0] = liste des premières coordonnées des vecteurs de sol
                                                # _ : _ intervalle des valeurs de la liste = sous liste
plt.legend(loc='best')                          #légende au meilleur endroit
plt.xlabel('t (s)')                             #légende axe x
plt.ylabel('tension en V')                            
plt.grid()
plt.show()


# portrait de phase
# plt.figure(2)
# plt.plot(sol[:, 0], sol[:, 1],  label= 'm =' + str(m))
# plt.xlabel('theta en rad')
# plt.legend(loc='lower right') 
# plt.ylabel('thetapoint en rad/s')
# plt.grid()
# plt.show()

